function out = quaterode_forward_model(mp, d, p, consts, dFlag)

if nargin < 5
    dFlag = 0;
else
end

% This script calculates Be-10 concentrations over a set expsoure/burial
% history. During exposure periods, cosmogenic-nuclides are produced under
% some steady erosion rate. During burial periods, no production takes
% place but subglacial erosion does. 

% St scaling and Balco muons. Start with simple case. SLHL, 3 stage
% exposure history = [exposure burial exposure]. erosion rate = [subaerial subglacial subaerial].

% Allie Balter-Kennedy, February 2024

%% Update model parameters

add_t = [(mp.hp.fixy - d.deglac_t) d.deglac_t];
t_eb = [mp.hp.t_eb add_t];
add_mask = [0 1]; % careful here, there could be situations in which this doesn't work?
p_mask = [mp.hp.p_mask add_mask];
add_erate = [mp.ep.e_rate(find(mp.hp.p_mask == 0, 1, 'last')) mp.ep.e_rate(find(mp.hp.p_mask == 1, 1, 'last'))]; % erate comes in in g cm^-2
e_rate = [mp.ep.e_rate add_erate];

%% Finish calculating model parameters

e_tot = t_eb .* e_rate; % total erosion per timestep in g*cm^[-2]
z_start = mp.z_gcm' + fliplr(cumsum(fliplr(e_tot))); % depth at start of each timestep in g*cm^[-2]*yr^[-1]
z_end = z_start - e_tot; % depth at end of each timestep in g*cm^[-2]*yr^[-1]

if max(max(z_start)) > max(consts.prezg)
    error('Start depth exceeds depth of pre-calculated muon production rates') %% need to revisit this - shouldn't be a problem right now, but will be to run for all of greenland.
else
end

% could make all NaNs in muon zero, but that's misleading. see Greg's use
% of Lbot... better approach.

%% Initialize

% Be-10
N10sp_new = zeros(size(mp.z_gcm))';
N10sp_old = N10sp_new;

N10_mu = zeros(length(mp.z_gcm), 1);
N10mu_new = N10sp_new;
N10mu_old = N10sp_new;

% Al-26
N26sp_new = N10sp_new;
N26sp_old = N10sp_new;

N26_mu = zeros(length(mp.z_gcm), 1);
N26mu_new = N10sp_new;
N26mu_old = N10sp_new;

% C-14
N14sp_new = N10sp_new;
N14sp_old = N10sp_new;

N14_mu = zeros(length(mp.z_gcm), 1);
N14mu_new = N10sp_new;
N14mu_old = N10sp_new;

%% run model

% should make this if statement for pmask so integral only runs on exposure

for a = 1:length(t_eb)
    
    if p_mask(a) == 1
    % Be-10 spallation
    N10_sp = ((p.P10sp .* exp(-(z_end(:, a)./consts.Lsp))) ./ (consts.l10 + (e_rate(a)./consts.Lsp))) .* ...
        (1 - exp(-(consts.l10 + (e_rate(a)./consts.Lsp)) .* t_eb(a)));

    % Al-26 spallation
    N26_sp = ((p.P26sp .* exp(-(z_end(:, a)./consts.Lsp))) ./ (consts.l26 + (e_rate(a)./consts.Lsp))) .* ...
        (1 - exp(-(consts.l26 + (e_rate(a)./consts.Lsp)) .* t_eb(a)));

    % C-14 spallation
    N14_sp = ((p.P14sp .* exp(-(z_end(:, a)./consts.Lsp))) ./ (consts.l14 + (e_rate(a)./consts.Lsp))) .* ...
        (1 - exp(-(consts.l14 + (e_rate(a)./consts.Lsp)) .* t_eb(a)));
    
    % muons
    for b = 1:length(mp.z_gcm)
        % Be-10
        N10_mu(b, :) = integral(@(t) exp(-consts.l10 .* t).* interp1(consts.prezg', p.premu10', (z_end(b,a) + (e_rate(a).*t))), 0, t_eb(a), 'reltol', 1e-3);
        
        % Al-26
        N26_mu(b, :) = integral(@(t) exp(-consts.l26 .* t).* interp1(consts.prezg', p.premu26', (z_end(b,a) + (e_rate(a).*t))), 0, t_eb(a), 'reltol', 1e-3);

        % C-14
        N14_mu(b, :) = integral(@(t) exp(-consts.l14 .* t).* interp1(consts.prezg', p.premu14', (z_end(b,a) + (e_rate(a).*t))), 0, t_eb(a), 'reltol', 1e-3);
    end
    
    elseif p_mask(a) == 0
        N10_sp = zeros(length(mp.z_gcm), 1);
        N10_mu = zeros(length(mp.z_gcm), 1);

        N26_sp = zeros(length(mp.z_gcm), 1);
        N26_mu = zeros(length(mp.z_gcm), 1);

        N14_sp = zeros(length(mp.z_gcm), 1);
        N14_mu = zeros(length(mp.z_gcm), 1);
        
    else
        disp('p_mask is not 1 or 0')
    end

    % Be-10 update variables 
    N10sp_new = (N10sp_old .* exp(-consts.l10.*t_eb(a))) + N10_sp; 
    N10mu_new = (N10mu_old .* exp(-consts.l10.*t_eb(a))) + N10_mu;


    N10sp_old = N10sp_new;
    N10mu_old = N10mu_new;

    % Al-26 update variables 
    N26sp_new = (N26sp_old .* exp(-consts.l26.*t_eb(a))) + N26_sp; 
    N26mu_new = (N26mu_old .* exp(-consts.l26.*t_eb(a))) + N26_mu;

    N26sp_old = N26sp_new;
    N26mu_old = N26mu_new;

    % C-14 update variables 
    N14sp_new = (N14sp_old .* exp(-consts.l14.*t_eb(a))) + N14_sp; 
    N14mu_new = (N14mu_old .* exp(-consts.l14.*t_eb(a))) + N14_mu;

    N14sp_old = N14sp_new;
    N14mu_old = N14mu_new;

    if dFlag == 3
        N10tot = N10sp_new + N10mu_new;
        N26tot = N26sp_new + N26mu_new;
        N10accum = N10_sp + N10_mu;
        N26accum = N26_sp + N26_mu;

        N10_t(a) = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, N10tot, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);
        N26_t(a) = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, N26tot, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);
        
        N10_accum(a) = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, N10accum, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);
        N26_accum(a) = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, N26accum, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);
    else
        % do nothing here
    end

end

% Final results

% Be-10
pred.N10sp = N10sp_new;
pred.N10mu = N10mu_new;
pred.N10tot = N10sp_new + N10mu_new;

% Al-26
pred.N26sp = N26sp_new;
pred.N26mu = N26mu_new;
pred.N26tot = N26sp_new + N26mu_new;

% C-14
pred.N14sp = N14sp_new;
pred.N14mu = N14mu_new;
pred.N14tot = N14sp_new + N14mu_new;

% result.N10 = pred.N10tot(1);

% calculate predicted concentrations for each sample

result.N10p = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, pred.N10tot, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);
result.N26p = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, pred.N26tot, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);
result.N14p = trapz(0:0.01:d.thick_gcm, interp1(mp.z_gcm, pred.N14tot, 0:0.01:d.thick_gcm))./diff([0 d.thick_gcm]);

% calculate misfits

result.miss10 = (result.N10p - d.N10)./d.dN10;
result.miss26 = (result.N26p - d.N26)./d.dN26;
result.x2 = sum(result.miss10.^2) + sum(result.miss26.^2);

if dFlag ==3
    plot.N10_t = N10_t;
    plot.N26_t = N26_t;
    plot.N10_accum = N10_accum;
    plot.N26_accum = N26_accum;
    plot.z_start = z_start(1, :);
    plot.z_end = z_end(1, :);
    plot.t_eb = t_eb;
else 
    % no plot stuff
end

if dFlag == 0
    out = result.x2;
elseif dFlag == 1
    out = result;
elseif dFlag == 3 
    out = plot;
end

end














